#!/bin/bash

##############################################################################
# Switch between Apache and LiteSpeed Web Server under control panel environment
#
# @author LiteSpeed Technologies, Inc. (https://www.litespeedtech.com)
# @copyright (c) 2016-2020
##############################################################################

check_errs()
{
  if [ $1 -ne 0 ] ; then
    echo "[ERROR] ${2}" 
    exit $1
  fi
}

INST_USER=`id`
INST_UID=`expr "$INST_USER" : 'uid=\(.*\)(.*) gid=.*'`
INST_USER=`expr "$INST_USER" : 'uid=.*(\(.*\)) gid=.*'`

if [ $INST_UID -ne 0 ]; then
    check_errs 1 "Only root user can switch web server!"
fi

display_usage()
{
    cat <<EOF
cp_switch_ws.sh - Switch between Apache and LiteSpeed under control panel environments.
Currently supported control panels are cPanel, Plesk, DirectAdmin, and CentOS Web Panel.
CustomBuild 2.0 is required to be installed prior for DirectAdmin detection.

Usage: cp_switch_ws.sh apache|lsws
Param: targeted web server. Fixed value either "apache" or "lsws"

EOF
    exit 1
}

detect_control_panel()
{
    # detect cPanel
    if [ -d "/usr/local/cpanel/whostmgr" ] ; then
        CP="WHM"
        cp_init_var
        echo "Detected cPanel WHM Environment"
    # detect Plesk
    elif [ -e "/opt/psa/version" ] || [ -e "/usr/local/psa/version" ] ; then
        CP="PSA"
        cp_init_var
        echo "Detected Plesk Environment"
    # detect DirectAdmin
    elif [ -d "/usr/local/directadmin" ] ; then
        CP="DA"
        DA_DIR="/usr/local/directadmin/custombuild"

        echo "Detected DirectAdmin Environment"
    # detect CentOS Web Panel
    elif [ -d "/usr/local/cwpsrv" ] ; then
        CP="CWP"
        echo "Detected CentOS Web Panel Environment"
    else
        check_errs 1 "Cannot detect control panel environment. Only cPanel WHM, Plesk, DirectAdmin are checked for now."
    fi
}

cp_SetRunOnBoot()
{
    local chkconfigPath=`which chkconfig`

    $chkconfigPath lsws off
    $chkconfigPath httpd on

    return 0
}

cp_msg_exit()
{
    echo "$2"
    exit ${1}
}

cp_init_var()
{
    OS=`uname -s`
    
    if [ "x$OS" = "xFreeBSD" ] || [ "x$OS" = "xDarwin" ] ; then
        PS_CMD="ps -ax"
    else
        PS_CMD="ps -ef"
    fi

    if [ "x$OS" = "xFreeBSD" ] ; then
        LSWS_CTLCMD="/usr/local/etc/rc.d/lsws.sh"
        AP_CTLCMD="/usr/local/etc/rc.d/httpd"
    else
        LSWS_CTLCMD="/sbin/service lsws"
        AP_CTLCMD="/sbin/service httpd"
    fi

    LSWS_PIDFILE="/tmp/lshttpd/lshttpd.pid"
    CONF="${LSWSHOME}/conf/httpd_config.xml"
    AP_PROC="httpd"

    LSWS_SWITCH_FLAG="${LSWSHOME}/admin/tmp/.switched2lsws"


    if [ "$CP" = "PSA" ] && [ -e "/etc/debian_version" ] ; then
        LSWS_CTLCMD="/usr/sbin/service lsws"
        AP_CTLCMD="/usr/sbin/service apache2"
        AP_PROC="apache2"
        . "/etc/apache2/envvars" 2>/dev/null

        if [ $? != 0 ]; then
            . "/etc/apache2/envvars"
        fi

        AP_PIDFILE="$APACHE_PID_FILE"
    fi
}

cp_stop_lsws()
{
    if [ "$CP" = "WHM" ] ; then
        pkill wswatch.sh 2>/dev/null
    fi

    $LSWS_CTLCMD stop
    $AP_CTLCMD stop 2>&1 1>/dev/null

    RETRY=30
    LSPID=1

    while [ $RETRY -gt 0 ] && [ $LSPID -ne 0 ]
    do
        ((RETRY--))
        sleep 1
        cp_detect_lsws_pid
    done

    if [ $LSPID -eq 0 ] ; then
        pkill -9 lshttpd
        pkill -9 litespeed
        echo "LiteSpeed Stopped."
    else
        echo "LiteSpeed is still running. Fail to stop within 30 secs."
        echo "Will use pkill command to stop"

        pkill -9 lshttpd
        pkill -9 litespeed
        sleep 1

        cp_detect_lsws_pid

        if [ $LSPID -eq 0 ] ; then
            echo "LiteSpeed Stopped."
        else
            echo "LiteSpeed is still running. Fail to stop using kill command."
        fi
    fi

}

cp_detect_lsws_pid()
{
    LSPID=0

    if [ -f "$LSWS_PIDFILE" ] ; then
        FPID=`cat "$LSWS_PIDFILE"`
        
        if [ "x$FPID" != "x" ] ; then
            PL=`$PS_CMD | grep -w 'lshttpd\|litespeed' | grep -v grep | grep -w "$FPID"`
                
            if [ "x$PL" != "x" ] ; then
                LSPID=$FPID
            fi
        fi
    fi
}

#
# start LiteSpeed
#
cp_start_fake_apache()
{
    $AP_CTLCMD start 2>&1

    RETRY=30
    LSPID=0

    while [ $RETRY -gt 0 ] && [ $LSPID -eq 0 ]
    do
        ((RETRY--))
        sleep 1
        cp_detect_lsws_pid
    done
}

remove_bak_bak()
{
    if [ -f "$1$2" ]; then
        mv "$1$2" "$1$2_restore_done"
    fi
}

cpanel_apache_wrapper_helper()
{
    RESTORE=$1

    NEED_PROTECT=`/usr/local/cpanel/3rdparty/bin/perl -MCpanel::Config::Httpd::Perms -e'print Cpanel::Config::Httpd::Perms::webserver_runs_as_user();'`

    if [ $RESTORE -eq 1 ] ; then
        # restore Apache binary Files
        if [ -f "/usr/local/apache/bin/${AP_PROC}_ls_bak" ] ; then
            mv -f "/usr/local/apache/bin/${AP_PROC}_ls_bak" "/usr/local/apache/bin/${AP_PROC}"
        elif [ -f "/usr/local/apache/bin/${AP_PROC}_ls_bak_bak" ] ; then
            cp -f "/usr/local/apache/bin/${AP_PROC}_ls_bak_bak" "/usr/local/apache/bin/${AP_PROC}"
        fi
        remove_bak_bak "/usr/local/apache/bin/${AP_PROC}" "_ls_bak_bak"

        if [ -f "/usr/sbin/${AP_PROC}_ls_bak" ] ; then
            mv -f "/usr/sbin/${AP_PROC}_ls_bak" "/usr/sbin/${AP_PROC}"
        elif [ -f "/usr/sbin/${AP_PROC}_ls_bak_bak" ] ; then
            cp -f "/usr/sbin/${AP_PROC}_ls_bak_bak" "/usr/sbin/${AP_PROC}"
        fi
        remove_bak_bak "/usr/sbin/${AP_PROC}" "_ls_bak_bak"

        if [ -f "/scripts/restartsrv_httpd_ls_bak" ] ; then
            mv -f "/scripts/restartsrv_httpd_ls_bak" "/scripts/restartsrv_httpd"
        elif [ -f "/scripts/restartsrv_httpd_ls_bak_bak" ] ; then
            cp -f "/scripts/restartsrv_httpd_ls_bak_bak" "/scripts/restartsrv_httpd"
        fi
        remove_bak_bak "/scripts/restartsrv_httpd" "_ls_bak_bak"

        if [ $NEED_PROTECT -eq 1 ] ; then
            /scripts/enablefileprotect
            echo "Added fileprotect for Apache"
        fi

        if [ -f "/etc/cpanel/ea4/paths.conf" ]; then
            /bin/cp "/etc/cpanel/ea4/paths.conf" "/etc/cpanel/ea4/paths.conf.tmp"
            sed -e 's#/usr/local/lsws/bin/lswsctrl#/usr/sbin/apachectl#' </etc/cpanel/ea4/paths.conf.tmp >/etc/cpanel/ea4/paths.conf
        fi

        # restore rc file
        if [ "$SYSTEMDDIR" != "" ]; then
            if [  -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" ]  ; then
                mv -f "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" "${SYSTEMDDIR}/${AP_PROC}.service"
            elif [  -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak_bak" ]  ; then
                cp -f "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak_bak" "${SYSTEMDDIR}/${AP_PROC}.service"
            fi
            remove_bak_bak "${SYSTEMDDIR}/${AP_PROC}.service" ".ls_bak_bak"
            systemctl daemon-reload
        fi

        if [ "$INIT_DIR" != "" ]; then
            if [ -e "${INIT_DIR}/${AP_PROC}.ls_bak" ] ; then
                mv -f "${INIT_DIR}/${AP_PROC}.ls_bak" "${INIT_DIR}/${AP_PROC}"
            elif [ -e "${INIT_DIR}/${AP_PROC}.ls_bak_bak" ] ; then
                cp -f "${INIT_DIR}/${AP_PROC}.ls_bak_bak" "${INIT_DIR}/${AP_PROC}"
            fi
            remove_bak_bak "${INIT_DIR}/${AP_PROC}" ".ls_bak_bak"
        fi

        if [ -f "$LSWS_SWITCH_FLAG" ] ; then
            /bin/rm -f "$LSWS_SWITCH_FLAG"
        fi

    else

        if [ $NEED_PROTECT -eq 1 ] ; then 
            /scripts/disablefileprotect
            echo "fileprotect removed, not needed by LiteSpeed"
        fi

        if [ -f "/etc/cpanel/ea4/paths.conf" ]; then
            /bin/cp "/etc/cpanel/ea4/paths.conf" "/etc/cpanel/ea4/paths.conf.tmp"
            sed -e 's#/usr/sbin/apachectl#/usr/local/lsws/bin/lswsctrl#' </etc/cpanel/ea4/paths.conf.tmp >/etc/cpanel/ea4/paths.conf
        fi

        # add rc wrapper
        if [ "$SYSTEMDDIR" != "" ]  ; then
            # copy the correct file again to avoid wrong data
            ${LSWSHOME}/admin/misc/rc-inst.sh

            if  [ ! -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" ] ; then
                mv -f "${SYSTEMDDIR}/${AP_PROC}.service" "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak"
            fi

            ln -sf "${SYSTEMDDIR}/lshttpd.service" "${SYSTEMDDIR}/${AP_PROC}.service"
            systemctl daemon-reload
        fi

        if [ "$INIT_DIR" != "" ] ; then
            
            if [ ! -e "${INIT_DIR}/${AP_PROC}.ls_bak" ] ; then
                mv -f "${INIT_DIR}/${AP_PROC}" "${INIT_DIR}/${AP_PROC}.ls_bak"
            fi

            ln -sf ./lsws "${INIT_DIR}/${AP_PROC}"
        fi
        
        # set flag
        touch "$LSWS_SWITCH_FLAG"
    fi
}

plesk_apache_wrapper_helper()
{
    RESTORE=$1

    if [ $RESTORE -eq 1 ] ; then
        # restore Apache binary Files
        if [ -f "/usr/sbin/${AP_PROC}_ls_bak" ] ; then
            mv -f "/usr/sbin/${AP_PROC}_ls_bak" "/usr/sbin/${AP_PROC}"
        elif [ -f "/usr/sbin/${AP_PROC}_ls_bak_bak" ] ; then
            cp -f "/usr/sbin/${AP_PROC}_ls_bak_bak" "/usr/sbin/${AP_PROC}"
        fi
        remove_bak_bak "/usr/sbin/${AP_PROC}" "_ls_bak_bak"

        # restore rc file
        if [ "$SYSTEMDDIR" != "" ]; then
            if [ -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" ]  ; then
                mv -f "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" "${SYSTEMDDIR}/${AP_PROC}.service"
            elif [ -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak_bak" ]  ; then
                cp -f "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak_bak" "${SYSTEMDDIR}/${AP_PROC}.service"
            fi
            remove_bak_bak "${SYSTEMDDIR}/${AP_PROC}.service" ".ls_bak_bak"

            if [ -d "${SYSTEMDDIR}/lshttpd.service.d" ]; then
                mv -f "${SYSTEMDDIR}/lshttpd.service.d" "${SYSTEMDDIR}/${AP_PROC}.service.d"
            fi

            systemctl daemon-reload
            systemctl disable lshttpd
            systemctl enable ${AP_PROC}
        fi

        if [ "$INIT_DIR" != "" ] ; then
            if [ -e "${INIT_DIR}/${AP_PROC}.ls_bak" ]  ; then
                mv -f "${INIT_DIR}/${AP_PROC}.ls_bak" "${INIT_DIR}/${AP_PROC}"
            elif [ -e "${INIT_DIR}/${AP_PROC}.ls_bak_bak" ]  ; then
                cp -f "${INIT_DIR}/${AP_PROC}.ls_bak_bak" "${INIT_DIR}/${AP_PROC}"
            fi
            remove_bak_bak "${INIT_DIR}/${AP_PROC}.service" ".ls_bak_bak"
            if [ ${SYSTEMDEXIST} -eq 1 ] ; then
                systemctl daemon-reload
            fi
        fi

        if [ -f "$LSWS_SWITCH_FLAG" ] ; then
            /bin/rm -f "$LSWS_SWITCH_FLAG"
        fi
    else
        # add rc wrapper
        if [ "$SYSTEMDDIR" != "" ]  ; then
            
            if  [ ! -e "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak" ] ; then
                systemctl disable ${AP_PROC}
                systemctl enable lshttpd
                mv -f "${SYSTEMDDIR}/${AP_PROC}.service" "${SYSTEMDDIR}/${AP_PROC}.service.ls_bak"
            fi
            
            ln -sf "${SYSTEMDDIR}/lshttpd.service" "${SYSTEMDDIR}/${AP_PROC}.service"

            if [ -d "${SYSTEMDDIR}/${AP_PROC}.service.d" ]; then
                mv -f "${SYSTEMDDIR}/${AP_PROC}.service.d" "${SYSTEMDDIR}/lshttpd.service.d"
            fi

            systemctl daemon-reload
        fi

        if [ "$INIT_DIR" != "" ] ; then
            
            if [ ! -e "${INIT_DIR}/${AP_PROC}.ls_bak" ] ; then
                mv -f "${INIT_DIR}/${AP_PROC}" "${INIT_DIR}/${AP_PROC}.ls_bak"
            fi
            
            cp "${INIT_DIR}/lsws" "${INIT_DIR}/${AP_PROC}"
            #ln -sf ./lsws "${INIT_DIR}/${AP_PROC}"
            
            if [ ${SYSTEMDEXIST} -eq 1 ] ; then
                systemctl daemon-reload
            fi
        fi
        
        # set flag
        touch "$LSWS_SWITCH_FLAG"
    fi
}

cp_apache_wrapper()
{
    RESTORE=$1

    INIT_DIR=""
    for path in "/etc/init.d" "/etc/rc.d/init.d"
    do
        if [ "$INIT_DIR" = "" ] \
                && { [ -e "${path}/${AP_PROC}" ] || [ -e "${path}/${AP_PROC}.ls_bak" ] ;}
        then

            INIT_DIR="$path"
        fi
    done

    # use systemd if possible, need to use same method as apache
    SYSTEMDDIR=""
    
    for path in "/etc/systemd/system" "/usr/lib/systemd/system" "/lib/systemd/system"
    do
        if [ "$SYSTEMDDIR" = "" ] \
              && [ -d "$path" ] \
              && [ -e "${path}/${AP_PROC}.service" ] ; then

            SYSTEMDDIR="$path"
        fi
    done

    if [ "$CP" = "WHM" ] ; then
        cpanel_apache_wrapper_helper $RESTORE
    elif [ "$CP" = "PSA" ] ; then
        plesk_apache_wrapper_helper $RESTORE
    fi
}

cp_detect_ap_pid()
{
    APPID=0

    FPID=`$PS_CMD | grep -w "${AP_PROC} \|${AP_PROC}$\|^${AP_PROC}$" \
            | grep -v "lscgid\|litespeed\|lshttpd\|grep\|/usr/bin/logger" | grep -w "root" \
            | awk '{print $2}'`
            
    if [ "x$FPID" != "x" ] ; then
        APPID=$FPID
    fi
}

cp_stop_apache()
{
    $AP_CTLCMD stop

    RETRY=30
    APPID=1

    while [ $RETRY -gt 0 ] && [ $APPID -ne 0 ]
    do
        ((RETRY--))
        sleep 1
        cp_detect_ap_pid
    done

   if [ $APPID -eq 0 ] ; then
        echo "Apache Stopped."
    else
        echo "Apache is still running. Fail to stop within 30 secs."
        echo "Will use pkill command to stop"

        pkill -9 $AP_PROC
        sleep 1

        cp_detect_ap_pid

        if [ $APPID -eq 0 ] ; then
            echo "Apache Stopped."
        else
            echo "Apache is still running. Fail to stop using kill command."
        fi
    fi
}

cp_change_port_offset()
{
    PORT_OFFSET=$1

    if [ ! -f "$CONF" ] ; then
        check_errs 1 "${0}: invalid conf file directory!"
    fi

    /bin/cp -f "$CONF" "${CONF}.orig"

    sed -e "s/<apachePortOffset>.*<\/apachePortOffset>/<apachePortOffset>${PORT_OFFSET}<\/apachePortOffset>/" "${CONF}.orig" > "$CONF"

    check_errs $? "${0}: sed command error, please try to modify apache port offset manually from config file ${CONF}"
    
    /bin/rm -f "${CONF}.orig"
}

cp_SwitchToLiteSpeed()
{
    cp_change_port_offset 0
    
    cp_stop_apache
    
    if [ "$CP" = "WHM" ] ; then
        #sleep 8 
        pkill -9 httpd
    elif [ "$CP" = "PSA" ] ; then
        cp_detect_lsws_pid
    fi
    
    cp_apache_wrapper 0

    if [ "$CP" = "WHM" ] ; then
        /usr/local/cpanel/bin/whmapi1 configureservice service=httpd enabled=1 monitored=1
    fi

    cp_start_fake_apache

    if [ $LSPID -gt 0 ] ; then
    
        if [ "$CP" = "WHM" ] ; then
            pkill wswatch.sh 2>/dev/null
            cd /usr/local/lsws/logs
            nohup ../bin/wswatch.sh </dev/null >/dev/null 2>&1 &
        fi

        cp_msg_exit 0 "LiteSpeed started successfully."
    fi

    echo "LiteSpeed is not up within 30 secs, try again by removing /tmp/lshttpd/."

    pkill -9 lshttpd
    pkill -9 litespeed
    /bin/rm -rf "/tmp/lshttpd"

    cp_start_fake_apache

    # wrapper needs to run after lsws start

    if [ $LSPID -gt 0 ] ; then
        
        if [ "$CP" = "WHM" ] ; then
            pkill wswatch.sh 2>/dev/null
            cd /usr/local/lsws/logs
            nohup ../bin/wswatch.sh 2>/dev/null &
        fi

        cp_msg_exit 0 "LiteSpeed started successfully."
    else
        cp_apache_wrapper 1
        check_errs 1 "LiteSpeed is not up within 60 secs, please check the error log and try again."
    fi
}

cp_SwitchToApache()
{
    if [ "$CP" = "WHM" ] ; then
        pkill wswatch.sh 2>/dev/null
    fi

    cp_detect_lsws_pid
    
    if [ $LSPID -gt 0 ] ; then
        echo "LiteSpeed is running, stop it first."
        
        cp_stop_lsws

        if [ $LSPID -gt 0 ] ; then
            check_errs 1 "Abort."
        fi
    fi

    cp_detect_ap_pid
    
    if [ $APPID -gt 0 ] ; then
        # if running, stop first
        cp_stop_apache
    fi

    # restore Apache Files, wrapper needs to run before Apache start
    cp_apache_wrapper 1    

    # wait 1 sec before start apache
    sleep 1

    if [ "$CP" = "WHM" ] ; then
        /usr/local/cpanel/bin/whmapi1 configureservice service=httpd enabled=1 monitored=1

        #
        # start Apache
        #
        # does not change the memory limits, does not work well.
        # $AP_CTLCMD start
        #
        # call a cPanel script
        /scripts/restartsrv_httpd
    
    elif [ "$CP" = "PSA" ] ; then
        $AP_CTLCMD start 2>&1
    fi

    RETRY=30
    APPID=0

    while [ $RETRY -gt 0 ] && [ $APPID -eq 0 ]
    do
        cp_detect_ap_pid
        ((RETRY--))

        sleep 1
    done
     
    if [ $APPID -ne 0 ] ; then
        cp_msg_exit 0 "Apache started successfully."
    else
        check_errs 1 "Apache is not up within 30 secs. Please check the log file."
    fi
}

switch_cp()
{
    if [ "x$1" = "xapache" ] ; then
        cp_SwitchToApache
        cp_SetRunOnBoot
    elif [ "x$1" = "xlsws" ] ; then
        cp_SwitchToLiteSpeed
        cp_SetRunOnBoot
    fi
}

switch_da()
{
    cd ${DA_DIR}

    pmod1=`grep "php1_mode" /usr/local/directadmin/custombuild/options.conf | cut -d = -f 2 | xargs`
    pmod2=`grep "php2_mode" /usr/local/directadmin/custombuild/options.conf | cut -d = -f 2 | xargs`

    if [ "x$1" = "xapache" ] ; then
        pkill wswatch.sh
        ./build set webserver apache
        OS_NAME=$(cat /etc/redhat-release | awk '{print $1}')
        if [ ${OS_NAME} == 'CloudLinux' ] ; then
            ./build set php1_mode lsphp
            ./build set php2_mode lsphp
        else
            ./build set php1_mode php-fpm
            ./build set php2_mode php-fpm
        fi
        ./build set mod_ruid2 no
        ./build apache
    else
        ./build set webserver litespeed
        if [ "x$pmod1" != "xlsphp" ] ; then
            ./build set php1_mode lsphp
            if [ "x$pmod2" != "xlsphp" ] ; then
                ./build set php2_mode lsphp
            fi
        fi
        ./build litespeed
    fi
}

switch_cwp() 
{
    if [ "x$1" = "xapache" ] ; then
        pkill wswatch.sh
        stop_running_ws
        echo "Starting Apache..."
        if [ -e "/usr/local/apache/bin/httpd_ls_bak" ] ; then
            mv /usr/local/apache/bin/httpd_ls_bak /usr/local/apache/bin/httpd
        fi
        service_manager "start" "httpd"
        echo "Apache has been started successfully."
    elif [ "x$1" = "xlsws" ] ; then
        stop_running_ws
        echo "Starting LiteSpeed Enterprise..."
        service_manager "start" "lsws"
        echo "LiteSpeed Enterprise has been started successfully."
    fi
}

service_manager() 
{
    if [ -d "/etc/systemd" ] ; then
        systemctl "$1" "$2" > /dev/null 2>&1
    else
        service "$2" "$1" > /dev/null 2>&1
    fi
}

stop_running_ws()
{
    local AP_PIDFILE=/usr/local/apache/logs/httpd.pid
    local LSWS_PIDFILE=/tmp/lshttpd/lshttpd.pid

    if [ -e "$AP_PIDFILE" ] ; then
        local AP_PID=$(cat $AP_PIDFILE)
    fi
    if [ -e "$LSWS_PIDFILE" ] ; then
        local LSWS_PID=$(cat $LSWS_PIDFILE)
    fi

    if [ -e "$LSWS_PIDFILE" ] && [ "$(ps -ef | grep -w 'lshttpd\|litespeed' | grep -v grep | grep -w $LSWS_PID)" != "" ] || [ "$(ps -ef | grep -w 'lshttpd\|litespeed' | grep -v grep)" != "" ] ; then
        echo "Stopping LiteSpeed Enterprise..."
        service_manager "stop" "lsws"
        sleep 1

        RETRY=30
        CHECK=$(ps -ef | grep -w 'lshttpd\|litespeed' | grep -v grep)

        while [ "$RETRY" -gt 0 ] && [ "$CHECK" != "" ] ; do
            RETRY=$(($RETRY-1))
            sleep 1
            CHECK=$(ps -ef | grep -w 'lshttpd\|litespeed' | grep -v grep)
        done

        pkill -9 lsphp > /dev/null 2>&1

        if [ "$CHECK" != "" ] ; then
            pkill -9 litespeed > /dev/null 2>&1
            pkill -9 lshttpd > /dev/null 2>&1
            pkill -9 lscgid > /dev/null 2>&1
            sleep 1
        fi

        if [ "$(ps -ef | grep -w 'lshttpd\|litespeed' | grep -v grep)" != "" ] ; then
            echo "LiteSpeed Enterprise could not be stopped."
            exit 1
        else
            echo "LiteSpeed Enterprise has been stopped successfully."
        fi
    fi

    if [ -e "$AP_PIDFILE" ] && [ "$(ps -ef | grep -w $AP_PID | grep httpd | grep -v grep | grep -v lscgid)" != "" ] || [ "$(ps -ef | grep httpd | grep -v grep | grep -v lscgid)" != "" ] ; then
        echo "Stopping Apache..."
        service_manager "stop" "httpd"
        sleep 1

        RETRY=30
        CHECK=$(ps -ef | grep httpd | grep -v grep | grep -v lscgid)

        while [ "$RETRY" -gt 0 ] && [ "$CHECK" != "" ] ; do
            ((RETRY--))
            sleep 1
            CHECK=$(ps -ef | grep httpd | grep -v grep | grep -v lscgid)
        done

        if [ "$CHECK" != "" ] ; then
            pkill -9 httpd > /dev/null 2>&1
            sleep 1
        fi

        if [ "$(ps -ef | grep httpd | grep -v grep | grep -v lscgid)" != "" ] ; then
            echo "Apache could not be stopped."
            exit 1
        else
            echo "Apache killed successfully."
        fi
    fi
}

if [ $# -ne 1 ] ; then
    echo "Illegal parameters = $# !"
    display_usage
fi

# set LSWSHOME
cd `dirname "$0"`
CURDIR=`pwd`
LSWSHOME=`dirname $CURDIR`
LSWSHOME=`dirname $LSWSHOME`

if [ "x$1" != "xapache" ] && [ "x$1" != "xlsws" ] ; then
    display_usage
fi

detect_control_panel

if [ "$CP" = "WHM" ] || [ "$CP" = "PSA" ] ; then
    switch_cp $1
elif [ "$CP" = "DA" ] ; then
    switch_da $1
elif [ "$CP" = "CWP" ] ; then
    switch_cwp $1
fi
